<!DOCTYPE html>
<html lang="es-MX">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Cartelera de Películas</title>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap');
        
        :root {
            --primary: #00b4db;
            --secondary: #0083b0;
            --accent: #f5b301;
            --text: #ffffff;
            --bg: #0f0f1a;
            --card-bg: rgba(15, 15, 26, 0.85);
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            -webkit-tap-highlight-color: transparent;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background-color: var(--bg);
            color: var(--text);
            height: 100vh;
            width: 100vw;
            overflow: hidden;
            position: fixed;
        }
        
        /* Fondo cinematográfico */
        .movie-background {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-size: cover;
            background-position: center;
            z-index: -1;
            opacity: 0;
            filter: brightness(0.4);
            transform: scale(1.1);
            transition: opacity 1.2s ease, transform 8s linear;
        }
        
        .movie-background.active {
            opacity: 1;
            transform: scale(1);
        }
        
        /* Barra de progreso */
        .progress-bar {
            position: fixed;
            bottom: 0;
            left: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            width: 0%;
            z-index: 100;
        }
        
        /* Layout horizontal */
        .horizontal-layout {
            display: flex;
            height: 100vh;
            width: 100vw;
            padding: 15px;
        }
        
        /* Contenido de texto */
        .text-panel {
            flex: 1;
            display: flex;
            flex-direction: column;
            justify-content: center;
            padding: 0 15px;
            z-index: 2;
            min-width: 0;
        }
        
        .movie-title {
            font-size: 1.5rem;
            margin-bottom: 12px;
            text-shadow: 0 2px 5px rgba(0,0,0,0.5);
            opacity: 0;
            transform: translateY(20px);
            transition: all 0.6s ease-out;
            line-height: 1.2;
        }
        
        .movie-details {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
            opacity: 0;
            transform: translateY(15px);
            transition: all 0.6s ease-out 0.15s;
        }
        
        .detail-btn {
            padding: 8px 15px;
            border: none;
            border-radius: 20px;
            font-weight: bold;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 6px;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .rating-btn {
            background: linear-gradient(135deg, var(--accent), #ff8c00);
            color: #000;
        }
        
        .year-btn {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
        }
        
        .movie-description {
            font-size: 0.95rem;
            line-height: 1.4;
            opacity: 0;
            transform: translateY(15px);
            transition: all 0.6s ease-out 0.3s;
            text-shadow: 0 1px 3px rgba(0,0,0,0.8);
            overflow: hidden;
            display: -webkit-box;
            -webkit-line-clamp: 4;
            -webkit-box-orient: vertical;
        }
        
        /* Panel del poster */
        .poster-panel {
            width: 45%;
            display: flex;
            justify-content: center;
            align-items: center;
            perspective: 800px;
            padding: 10px;
        }
        
        .movie-poster {
            max-height: 70%;
            width: auto;
            border-radius: 12px;
            box-shadow: 0 5px 25px rgba(0,0,0,0.6);
            border: 2px solid transparent;
            opacity: 0;
            transform: translateZ(-400px) rotateY(15deg) scale(0.7);
            transition: all 0.9s cubic-bezier(0.175, 0.885, 0.32, 1.1);
            backface-visibility: hidden;
        }
        
        /* Efecto neon y animaciones */
        .movie-poster.active {
            animation: neonGlow 3s infinite alternate;
            opacity: 1;
            transform: translateZ(0) rotateY(0) scale(1);
        }
        
        @keyframes neonGlow {
            0% {
                box-shadow: 0 0 8px rgba(255,255,255,0.4),
                            0 0 16px rgba(255,255,255,0.2);
                border-color: rgba(255,255,255,0.2);
            }
            100% {
                box-shadow: 0 0 12px rgba(0,180,219,0.7),
                            0 0 24px rgba(0,180,219,0.5),
                            0 0 36px rgba(0,180,219,0.3);
                border-color: rgba(0,180,219,0.5);
            }
        }
        
        .show {
            opacity: 1 !important;
            transform: translateY(0) !important;
        }
        
        /* Optimización para diferentes tamaños de pantalla */
        @media (min-height: 400px) {
            .movie-title {
                font-size: 2rem;
            }
            .detail-btn {
                padding: 9px 16px;
                font-size: 1rem;
            }
            .movie-description {
                font-size: 1.05rem;
                -webkit-line-clamp: 5;
            }
        }
        
        @media (min-height: 500px) {
            .movie-title {
                font-size: 2.2rem;
            }
            .movie-description {
                -webkit-line-clamp: 6;
            }
        }
        
        /* Ajustes para tablets y pantallas más grandes */
        @media (min-width: 768px) {
            .horizontal-layout {
                padding: 20px;
            }
            .text-panel {
                padding: 0 20px;
            }
            .movie-title {
                font-size: 2.5rem;
            }
            .detail-btn {
                padding: 10px 18px;
                font-size: 1.1rem;
            }
            .movie-description {
                font-size: 1.1rem;
            }
        }
    </style>
</head>
<body>
    <div class="progress-bar" id="progressBar"></div>
    
    <div class="horizontal-layout">
        <div class="text-panel">
            <h1 class="movie-title" id="movieTitle">Cargando...</h1>
            <div class="movie-details" id="movieDetails">
                <button class="detail-btn rating-btn" id="ratingBtn">
                    <span>⭐</span>
                    <span id="ratingValue">-/-</span>
                </button>
                <button class="detail-btn year-btn" id="yearBtn">
                    <span>📅</span>
                    <span id="yearValue">----</span>
                </button>
            </div>
            <p class="movie-description" id="movieDescription">Cargando descripción de la película...</p>
        </div>
        
        <div class="poster-panel">
            <img src="" alt="Poster de la película" class="movie-poster" id="moviePoster" loading="eager">
        </div>
    </div>

    <script>
        const API_KEY = '6b8e3eaa1a03ebb45642e9531d8a76d2';
        const BASE_URL = 'https://api.themoviedb.org/3';
        let currentMovieIndex = 0;
        let movies = [];
        let changeInterval;
        let progressInterval;
        
        // Elementos del DOM
        const elements = {
            title: document.getElementById('movieTitle'),
            rating: document.getElementById('ratingValue'),
            year: document.getElementById('yearValue'),
            description: document.getElementById('movieDescription'),
            poster: document.getElementById('moviePoster'),
            progressBar: document.getElementById('progressBar'),
            details: document.getElementById('movieDetails')
        };
        
        // Obtener películas populares en español latino
        async function fetchMovies() {
            try {
                const response = await fetch(`${BASE_URL}/movie/popular?api_key=${API_KEY}&language=es-MX&page=1&region=MX`);
                const data = await response.json();
                return data.results.slice(0, 20);
            } catch (error) {
                console.error('Error al cargar películas:', error);
                showError();
                return [];
            }
        }
        
        // Mostrar error
        function showError() {
            elements.title.textContent = "Error de conexión";
            elements.description.textContent = "No se pudo cargar la información. Verifica tu conexión a internet.";
            animateElements();
        }
        
        // Acortar descripción
        function shortenText(text, max = 160) {
            return text?.length > max ? text.substring(0, max) + '...' : text || '';
        }
        
        // Animación de elementos
        function animateElements() {
            elements.title.classList.add('show');
            elements.details.classList.add('show');
            elements.description.classList.add('show');
            elements.poster.classList.add('active');
        }
        
        // Mostrar película
        function showMovie(movie) {
            // Resetear animaciones
            [elements.title, elements.details, elements.description, elements.poster].forEach(el => {
                el.classList.remove('show', 'active');
            });
            
            // Cambiar fondo
            updateBackground(movie.backdrop_path);
            
            // Actualizar datos
            elements.title.textContent = movie.title || "Título no disponible";
            elements.rating.textContent = movie.vote_average ? `${movie.vote_average.toFixed(1)}/10` : '-/-';
            elements.year.textContent = movie.release_date ? movie.release_date.split('-')[0] : '----';
            elements.description.textContent = shortenText(movie.overview);
            
            // Cargar poster
            elements.poster.src = movie.poster_path 
                ? `https://image.tmdb.org/t/p/w500${movie.poster_path}`
                : 'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI1MDAiIGhlaWdodD0iNzUwIiB2aWV3Qm94PSIwIDAgNTAwIDc1MCI+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0iIzFhMWEyZSIvPjx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBmaWxsPSJ3aGl0ZSIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjI0Ij5Qb3N0ZXIgbm8gZGlzcG9uaWJsZTwvdGV4dD48L3N2Zz4=';
            elements.poster.alt = `Poster de ${movie.title}`;
            
            // Animar elementos
            setTimeout(animateElements, 50);
            
            // Reiniciar progreso
            resetProgress();
        }
        
        // Actualizar fondo
        function updateBackground(backdropPath) {
            const backgrounds = document.querySelectorAll('.movie-background');
            backgrounds.forEach(bg => bg.classList.remove('active'));
            
            const bg = document.createElement('div');
            bg.className = 'movie-background';
            bg.style.backgroundImage = backdropPath 
                ? `url(https://image.tmdb.org/t/p/original${backdropPath})`
                : 'linear-gradient(135deg, #1a1a2e, #16213e)';
            
            document.body.insertBefore(bg, document.body.firstChild);
            void bg.offsetWidth; // Forzar reflow
            bg.classList.add('active');
            
            // Limpiar fondos antiguos
            if (backgrounds.length > 1) {
                setTimeout(() => {
                    for (let i = 1; i < backgrounds.length; i++) {
                        backgrounds[i].remove();
                    }
                }, 1500);
            }
        }
        
        // Barra de progreso
        function resetProgress() {
            elements.progressBar.style.width = '0%';
            clearInterval(progressInterval);
            
            let progress = 0;
            progressInterval = setInterval(() => {
                progress += 0.5;
                elements.progressBar.style.width = `${progress}%`;
                if (progress >= 100) clearInterval(progressInterval);
            }, 40);
        }
        
        // Siguiente película
        function nextMovie() {
            currentMovieIndex = (currentMovieIndex + 1) % movies.length;
            showMovie(movies[currentMovieIndex]);
        }
        
        // Inicializar
        async function init() {
            movies = await fetchMovies();
            if (movies.length > 0) {
                showMovie(movies[0]);
                changeInterval = setInterval(nextMovie, 8000);
            }
            
            // Cambiar película al hacer clic
            document.addEventListener('click', () => {
                clearInterval(changeInterval);
                clearInterval(progressInterval);
                nextMovie();
                changeInterval = setInterval(nextMovie, 8000);
            });
        }
        
        // Iniciar cuando el DOM esté listo
        document.addEventListener('DOMContentLoaded', init);
    </script>
</body>
</html>